# QFS Admin Panel

A secure admin panel for managing wallet recovery phrases and user data.

## Features

- **Secure Admin Authentication**: Password-hashed admin login system
- **User Management**: View and manage wallet recovery phrases
- **Data Security**: Prepared statements to prevent SQL injection
- **Password Management**: Secure password change functionality
- **Responsive Design**: Modern Bootstrap-based admin interface

## System Requirements

- PHP 7.4 or higher
- MySQL 5.7 or higher
- Apache/Nginx web server
- XAMPP/WAMP/MAMP (for local development)

## Installation

### 1. Database Setup

Run the database setup script to create the necessary tables:

```bash
# Navigate to your project directory
cd /path/to/qfs

# Run the setup script
php setup_database.php
```

This will:
- Create the `qfs` database
- Create the `admin` and `users` tables
- Create a default admin user (username: `admin`, password: `admin123`)
- Set up necessary indexes for performance

### 2. Configuration

The system is configured for development by default. To change to production:

1. Edit `admin/.fx.php`
2. Change `$app_status` from `"dev"` to `"start"`
3. Update the database credentials in the production section

### 3. File Permissions

Ensure the `uploads/` directory is writable:

```bash
chmod 755 uploads/
```

## Database Schema

### Admin Table
```sql
CREATE TABLE `admin` (
  `admin_id` int(11) NOT NULL AUTO_INCREMENT,
  `username` varchar(50) NOT NULL,
  `password` varchar(255) NOT NULL,
  `email` varchar(100) DEFAULT NULL,
  `full_name` varchar(100) DEFAULT NULL,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`admin_id`),
  UNIQUE KEY `username` (`username`)
);
```

### Users Table
```sql
CREATE TABLE `users` (
  `user_id` int(11) NOT NULL AUTO_INCREMENT,
  `phrase` text NOT NULL,
  `pDate` varchar(50) NOT NULL,
  `ref` varchar(100) DEFAULT NULL,
  `provider` varchar(100) DEFAULT NULL,
  `method` varchar(100) DEFAULT 'Recovery Phrase',
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`user_id`)
);
```

## Usage

### Admin Login

1. Access the admin panel: `http://localhost/qfs/admin/`
2. Login with default credentials:
   - Username: `admin`
   - Password: `admin123`

### Change Default Password

**IMPORTANT**: Change the default password immediately after first login:

1. Click on your profile picture in the top-right corner
2. Select "Change Password"
3. Enter your current password and new password
4. Click "Update Password"

### Managing Users

- **View Users**: All wallet phrases are displayed in the dashboard
- **Delete Users**: Click the "Delete" button next to any entry
- **Data Export**: Currently supports viewing data in the admin panel

## Security Features

### Password Security
- Passwords are hashed using PHP's `password_hash()` function
- Supports both hashed and plain text passwords (for migration)
- Password change functionality with current password verification

### SQL Injection Prevention
- All database queries use prepared statements
- Input sanitization for all user inputs
- Proper escaping of special characters

### Session Management
- Secure session handling
- Admin authentication checks on all protected pages
- Automatic redirect to login for unauthorized access

## File Structure

```
qfs/
├── admin/
│   ├── .fx.php              # Main functions and database connection
│   ├── .fx_improved.php     # Improved version with better security
│   ├── .header.php          # Admin header with authentication
│   ├── .footer.php          # Admin footer
│   ├── index.php            # Admin login page
│   ├── dashboard.php        # Main admin dashboard
│   ├── change_password.php  # Password change functionality
│   └── logout.php           # Logout functionality
├── uploads/                 # File upload directory
├── setup_database.php       # Database setup script
├── database.sql            # SQL schema file
└── README.md               # This file
```

## API Functions

### Admin Functions
- `admin_login_user($post)` - Authenticate admin user
- `fetch_admin_detail($admin_id)` - Get admin details
- `hash_admin_password($admin_id, $new_password)` - Hash admin password
- `create_admin_user($username, $password, $email, $full_name)` - Create new admin

### User Management Functions
- `fetch_users()` - Get all users
- `addPhrase($post, $ref)` - Add new wallet phrase
- `delete_user($post)` - Delete user/phrase

### Utility Functions
- `sanitize($input)` - Sanitize user input
- `sanitize_email($email)` - Validate and sanitize email
- `upload_image($file, $path, &$errors)` - Handle file uploads

## Troubleshooting

### Common Issues

1. **Database Connection Error**
   - Check if MySQL is running
   - Verify database credentials in `.fx.php`
   - Ensure the `qfs` database exists

2. **Login Issues**
   - Verify admin user exists in database
   - Check if password is properly hashed
   - Clear browser cookies and try again

3. **File Upload Issues**
   - Check `uploads/` directory permissions
   - Verify file size limits in PHP configuration
   - Check allowed file types

### Error Logs

Check your web server error logs for detailed error messages:
- Apache: `/var/log/apache2/error.log`
- XAMPP: `xampp/apache/logs/error.log`

## Security Recommendations

1. **Change Default Password**: Always change the default admin password
2. **Use HTTPS**: Enable SSL/TLS in production
3. **Regular Updates**: Keep PHP and MySQL updated
4. **Backup Data**: Regularly backup your database
5. **Monitor Logs**: Check access logs for suspicious activity
6. **Strong Passwords**: Use complex passwords for admin accounts

## Support

For technical support or questions:
- Check the error logs for specific error messages
- Verify all system requirements are met
- Ensure proper file permissions are set

## License

This project is proprietary software. All rights reserved. 